/**
(c) 2018 Contecon Software GmbH Author E. Schreiner

PicApport Add-on to analyse metadata of one or more selected photos
it supports options to filter the output by metadata-name and/or values.

**/

import com.drew.metadata.Directory;
import com.drew.metadata.Metadata;
import com.drew.metadata.Tag;
import com.drew.metadata.xmp.XmpDirectory

import de.contecon.picapport.groovy.IAddonContext;
import de.contecon.picapport.groovy.IAddonExecutionContext;
import de.contecon.picapport.groovy.IAddonFileToProcess;
import de.contecon.picapport.groovy.PhotoFileProcessor;
import net.essc.util.GenLog

import org.json.JSONObject;


class MetadataAnalyser extends PhotoFileProcessor {

public Map init(IAddonContext addonContext) {
            
	addonContext.getLogger().logMessage(" Addon loaded. Autor: E. Schreiner (c)2020 Contecon Software GmbH" );

	def meta =  [
                version:'1.1.0',
				functions: [
						   f1: [
							   name:      'Analyse metadata',
							   desc:      'Addon to analyse all available metadata with key and values',
							   permission:'pap:access:metadata',
							   
							   parameter: [
										  filterKey: [
											  type:  'text',
											  label: 'Filter by key',
                                              value: ''
											  ],
										  filterKeyIgnoreCase: [
											  type:   'checkbox',
											  label:  'Case insensitive',
											  value:  true, 										   
											  ],
										  filterValue: [
											  type:  'text',
											  label: 'Filter by value',
                                              value: '' 
											  ],
										  filterValueIgnoreCase: [
											  type:   'checkbox',
											  label:   'Case insensitive',
											  value:  true, 										   
											  ],
										  onlyShowResults: [
											  type: 'checkbox',
											  label: 'Only show photos with result' ,
                                              value:  true,
											  ],
                                          showXmpAsXml: [
                                              type: 'checkbox',
                                              label: 'Show XMP data as xml' ,
                                              value:  false,
                                              ],
                                          maxValueLength: [
                                            type: 'range',
                                            label: 'Max. length displayed' ,
                                            value: '100',
                                            min:   '0',
                                            max:   '1000',
                                            ],
									      ]		  
							   ]								
						   ],
				i18n: [
					  'de.f1.name':                       'Metadaten analysieren',
					  'de.f1.desc':                       'Analyse der für PicApport verfügbaren Metadaten',
					  'de.f1.filterKey.label':            'Filter Schlüssel',
					  'de.f1.filterKeyIgnoreCase.label':  'Ignoriere Groß-Kleinschreibung', 
					  'de.f1.filterValue.label':          'Filter nach Wert',  
					  'de.f1.filterValueIgnoreCase.label':'Ignoriere Groß-Kleinschreibung',								 
					  'de.f1.onlyShowResults.label':      'Zeige nur Fotos mit Ergebnis',
                      'de.f1.showXmpAsXml.label':         'Zeige komplette XMP-Struktur als XML',
                      'de.f1.maxValueLength.label':       'Maximal angezeige Textlänge',
					  ],						
				]
	}

public void start(IAddonContext addonContext, IAddonExecutionContext aec) {
    // remove this comment to display content of aec: println "Execute start: "+aec;
    
    aec.checkKeys  =false;
    aec.checkValues=false;
    
    aec.filterKey= aec.filterKey.trim()
    if(aec.filterKey.length() > 0) {
      aec.checkKeys = true;
      }
    
    aec.filterValue = aec.filterValue.trim()
    if(aec.filterValue.length() > 0) {
       aec.checkValues = true;
       }
      
    aec.setShowResults(true); 
    aec.valuesFound=false;
    }
  
public void processPhotoFile(IAddonContext addonContext, IAddonExecutionContext aec, IAddonFileToProcess fileToProcess) {
    def showPhotoInReport=!aec.onlyShowResults;
    def originalFile = fileToProcess.getOriginalFile();
    def baseAttrMap=[:] as LinkedHashMap; // Content og this map will always be visible below the image (Filename etc)
        baseAttrMap['Name']         = originalFile.getName();
        baseAttrMap['Path']         = originalFile.getParent();
        baseAttrMap['Length']       = originalFile.length();
        baseAttrMap['Last Modified']= new Date(originalFile.lastModified());
    
    def      dirMap   = [:] as TreeMap;
    Metadata metadata = fileToProcess.getMetadata();
    
    def      picapportMetadataMap = [:] as TreeMap; 
    
    // Loop over all directories found in metadata. See: https://github.com/drewnoakes/metadata-extractor
    for(Directory directory : metadata.getDirectories()) {
       def dirEntrys = [:] as TreeMap;
       
       if(directory instanceof XmpDirectory) {
          def xmpd = (XmpDirectory)directory; 
          def xmpProperties = ((XmpDirectory)directory).getXmpProperties();
          def xmpErrors     = [] as List;
             
          for (String key : xmpProperties.keySet()) {
              String value = xmpProperties.get(key).toString();
              if(testFilter(key, value, aec)) {
                dirEntrys[key] = cut(value, aec.maxValueLength as int);
                picapportMetadataMap[key] = value;
                }
              }
          Collection<Tag> tags = xmpd.getTags();
          for(Tag tag: tags) {
             if(testFilter(tag, aec)) {
               dirEntrys[tag.getTagName()] = cut(tag.getDescription(), aec.maxValueLength as int);
               }
             }
          Iterable<String> errors = xmpd.getErrors();
          for(String error: errors) {
             xmpErrors.add(error);
             }
          dirMap['xmp-errors']=xmpErrors;
          } else {
          for(Tag tag: directory.getTags()) {
              if(testFilter(tag, aec)) {
                dirEntrys[tag.getTagName()]=cut(tag.getDescription(), aec.maxValueLength as int);
                }
              }
          }  
       dirMap[directory.getName()]=dirEntrys;
       showPhotoInReport |= (!dirEntrys.isEmpty());
       }
     // Now add all loaded maps to our report generator
     if(showPhotoInReport) {  
       aec.getPhotoFileProcessorResultGenerator().addGroupData("PicApport fileinfo",baseAttrMap); // (Values below the image)
       dirMap.each { key, val ->
                   if(!val.isEmpty()) {
                     aec.valuesFound=true;
                     aec.getPhotoFileProcessorResultGenerator().addGroupData(key,val);
                     }
                   }
       if(aec.showXmpAsXml) {
         aec.getPhotoFileProcessorResultGenerator().addGroupData("XMP / XML"+(fileToProcess.isCreatedByPlugin()?" (shadow file)":""), fileToProcess.getXmpMetaAsXMLString());
		 if(fileToProcess.isCreatedByPlugin()) {
            if(fileToProcess.respondsTo("getXmpMetaAsXMLStringFromOriginalFile")) {
			   def xml=fileToProcess.getXmpMetaAsXMLStringFromOriginalFile();			    
		       aec.getPhotoFileProcessorResultGenerator().addGroupData("XMP / XML ("+originalFile.getName()+")", (null!=xml?xml:"XMP of this filetype not supported or not available"));
		      }
		    }
        } 		   
         
       // now print all PicApport Addon generated json data
       picapportMetadataMap.each { key, val ->
           try {
               def json =new JSONObject(val);
               aec.getPhotoFileProcessorResultGenerator().addGroupData("PicApport Addon: "+key,json);
               }
            catch(Exception e) {
               // Do nothing it's not a JSON metadata 
               }    
           }
       }         
    }

private boolean testFilter(Tag tag, IAddonExecutionContext aec) {
  return testFilter(tag.getTagName(), tag.getDescription(), aec);
}

private boolean testFilter(String key, String value, IAddonExecutionContext aec) {
  boolean result=true;
  
  if(null == value) {
    value = "";
    }

  if(null != key && aec.checkKeys) {
    if(aec.filterKeyIgnoreCase) {
      result &= key.containsIgnoreCase(aec.filterKey);
      } else {
      result &= key.contains(aec.filterKey);
      }
    }
    
  if(result && aec.checkValues) {
    if(aec.filterValueIgnoreCase) {
      result &= value.containsIgnoreCase(aec.filterValue);
      } else {
      result &= value.contains(aec.filterValue);
      }
    }
  return result;
}
    
private String cut(String s, int maxlen) {
    if(null != s && maxlen > 0 && s.length() > maxlen) {
      int r=s.length() - maxlen;
      s=s.take(maxlen)+" ...<${r}more>";
      }
    return s;  
}      
    
    
public void stop(IAddonContext addonContext, IAddonExecutionContext aec) {
    // nothing to do here	
    }
			
}