/**
(c) 2020 Contecon Software GmbH Author E. Schreiner

OSM Reverse geocoder

Doku:
https://nominatim.org/release-docs/develop/api/Reverse/


Sample URL's: 
https://nominatim.openstreetmap.org/reverse?format=json&lat=49.495829&lon=8.419093&addressdetails=1&accept-language=de
https://nominatim.openstreetmap.org/reverse?format=json&lat=49.638291&lon=8.280070&addressdetails=1&accept-language=de

https://www.geonames.org

**/

import org.json.JSONObject;

import de.contecon.picapport.IPhotoMetaData;
import de.contecon.picapport.groovy.IAddonContext;
import de.contecon.picapport.groovy.IAddonExecutionContext;
import de.contecon.picapport.groovy.IAddonFileToProcess;
import de.contecon.picapport.groovy.PhotoFileProcessor;

class OSMGeoReverseEncoder extends PhotoFileProcessor {

def VERSION       = "2.0.0";
def ADDON_TYPE_ID = "osm.org";  
  
def en_Title ='OpenStreetMap (OSM) Reverse geocoding';
def de_Title ='OpenStreetMap (OSM) Inverse Geokodierung';
  
def en_Options=['Write only when empty',  'Always overwrite',    'no Changes (just test)',        'Remove OpenStreetMap metadata',     'Show OpenStreetMap metadata of photos'];
def de_Options=['Schreibe nur wenn leer', 'Immer überschreiben', 'keine Änderungen (nur testen)', 'OpenStreetMap Metadaten entfernen', 'OpenStreetMap Daten der Fotos anzeigen'];
  
  
public Map init(IAddonContext addonContext) {
	addonContext.getLogger().logMessage(" Addon loaded Autor: E. Schreiner (c)2020 Contecon Software GmbH" );

	def meta =  [
                version:VERSION,
				functions: [
						   f1: [
							   name:       en_Title,
							   desc:       'Converts geographic coordinates (latitude, longitude)\nto a readable address or place name',
							   permission: 'pap:editmeta:photo',
							   
							   parameter: [
										  mode: [
											  type:   'select',
											  label:  'Overwrite',
											  options: en_Options,
                                              value:   '0'
											  ],
										  language: [
											  type: 'text',
											  label: 'language',
                                              value: addonContext.getConfigParAsString('language', System.getProperty("user.language")),
											  permission: 'pap:admin:addon:config'
											  ],
										  autoExecOnInsert: [
                                              type: 'checkbox',
                                              label: 'Automatic execution after insert new photos',
                                              value:  addonContext.getConfigParAsBoolean('autoExecOnInsert', false),
											  permission: 'pap:admin:addon:config'
                                              ],
                                          updateDefaults: [
                                              type:       'checkbox',
                                              label:      'Save current parameter values as defaults',
                                              value:      false,
                                              permission: 'pap:admin:addon:config'
                                              ],	  
                                          analyseResult: [
                                              type: 'checkbox',
                                              label: 'Analyse result',
                                              value: false 
                                              ]
										  ]											  
							   ]								   
						   ],
				i18n: [
					  'de.f1.name':                   de_Title,
					  'de.f1.desc':                   'Mittels Geokoordinaten werden textuelle Lokationsangaben gesucht,\nalso etwa Städtenamen, Straßennamen, etc...',
					  'de.f1.mode.label':             'Überschreiben',
					  'de.f1.mode.options':           de_Options,
                      'de.f1.language.label':         'Sprache',
					  'de.f1.autoExecOnInsert.label': 'Für neue Fotos automatisch ausführen',
					  'de.f1.updateDefaults.label':   'Aktuelle Parameter als Vorgabe speichern',
                      'de.f1.analyseResult.label':    'Analysiere Ergebnis',
					  ],						
				]
	}
	

private void handleDefaultParameterUpdate(IAddonContext addonContext, IAddonExecutionContext aec) {
  addonContext.putConfigPar("mode",            aec.mode);
  addonContext.putConfigPar("language",        aec.language);
  addonContext.putConfigPar("autoExecOnInsert",aec.autoExecOnInsert);
  addonContext.putConfigPar("analyseResult",   aec.analyseResult);
  addonContext.updateConfigFile();
  }	
  
public Map getAutoexecConfig(IAddonContext addonContext) {
  if(addonContext.getConfigParAsBoolean('autoExecOnInsert', false)) {
    return [id:            'f1',
	        mode:           '0', 
	        language:       addonContext.getConfigParAsString('language', System.getProperty("user.language")),
			updateDefaults: false,
			analyseResult:  false,
            ];
    }
  return null;
  }  
 
public void start(IAddonContext addonContext, IAddonExecutionContext aec) {

    addonContext.getLogger().logInfoMessage(" Add-on started for " + aec.getNumFilesToProcess() + " photos."); 

    aec.setShowResults(true); 
    aec.lastWebAccess=0;
    aec.currentPhoto=0; 
    aec.statisticAdded=0;
    aec.statisticReplaced=0;
    aec.statisticSkipped=0;
    aec.statisticNoGeoData=0;
    aec.statisticRemoved=0;
    aec.statisticQuery=0;

    def titleMap=["Addon Version":VERSION]; 

    if(aec.updateDefaults) {
      handleDefaultParameterUpdate(addonContext, aec);
      titleMap << ["Run-mode":addonContext.i18n("New parameter settings saved.",  ["de":"Neue Parametervorgaben gespeichert."])
                  ];
      aec.signalTermination();
	  } else {
	  titleMap << ["Run-mode":addonContext.i18n(en_Options[aec.mode as Integer],  ["de":de_Options[aec.mode as Integer]]),
                  (addonContext.i18n("Language",  ["de":"Sprache"])):aec.language 
                  ];
	  }

    aec.getPhotoFileProcessorResultGenerator().addGroupData(addonContext.i18n(en_Title,  ["de":de_Title] ), titleMap);
    }
    
public void processPhotoFile(IAddonContext addonContext, IAddonExecutionContext aec, IAddonFileToProcess fileToProcess) {
    aec.currentPhoto++;
    
    def mode = aec.mode as Integer;  
    def osmAlreadyExists = fileToProcess.hasAddonMetadata(ADDON_TYPE_ID) as boolean;
    def json = null as JSONObject;
  
    File originalFile = fileToProcess.getOriginalFile();
    def baseAttrs=[:] as LinkedHashMap; // Content of this map will always be visible below the image (Filename etc.)
        baseAttrs['Name']         = originalFile.getName();
        baseAttrs['Path']         = originalFile.getParent();
        baseAttrs['Length']       = originalFile.length();
        baseAttrs['Last Modified']= new Date(originalFile.lastModified());

    if(mode == 3 || mode == 4) { // Remove or Show "osm.org" Metadata call to osm.org is not required
      if(osmAlreadyExists) {
        // remove it
        if(mode == 3) { // remove
          fileToProcess.removeAddonMetadata(addonContext, ADDON_TYPE_ID);
          baseAttrs[ADDON_TYPE_ID+' Metadata']= "[[color:#8bc34a;]]Removed";
          aec.statisticRemoved++;
          } else {
          json = fileToProcess.getAddonMetadataAsJSON(ADDON_TYPE_ID);
          baseAttrs[ADDON_TYPE_ID+' Metadata from Photo']= "[[color:#8bc34a;]]Exists";
          aec.statisticQuery++;
          } 
        } else {
        // nothing to do
        baseAttrs[ADDON_TYPE_ID+' Metadata']= "[[color:#DD130E;]]"+addonContext.i18n("OpenStreetMap metadata not found in photo",  ["de":"Keine OpenStreetMap Metadaten im Foto gefunden"] );
        aec.statisticSkipped++;
        }
      if(aec.analyseResult || mode == 4) {
         aec.getPhotoFileProcessorResultGenerator().addGroupData("Result", baseAttrs);
         if(null != json) {
           aec.getPhotoFileProcessorResultGenerator().addGroupData("json", json);
           }   
         }
      return;
      }    
                  
    IPhotoMetaData photoMetadata=fileToProcess.getPhotoMetadata();
    if(!photoMetadata.hasGeoCoordinates()) {
      def noGeoCoordinates=addonContext.i18n("No GEO-Coordinates", ["de":"Keine Geo-Koordinaten"]);
      baseAttrs['Problem']="[[color:#DD130E;]]"+noGeoCoordinates;
      if(aec.analyseResult) {
         aec.getPhotoFileProcessorResultGenerator().addGroupData(noGeoCoordinates, baseAttrs);
         }
      aec.statisticNoGeoData++;
      return;       
      } 

    def saveJSON = (1 == mode) as boolean; // mode 1 = Always overwrite
    
    if(0 == mode) {
      if(osmAlreadyExists) {
        def alreadyExists=addonContext.i18n("OpenStreetMap-Data already loaded", ["de":"OpenStreetMap Daten bereits geladen"]);
        baseAttrs[ADDON_TYPE_ID+' Metadata']= "[[color:#8bc34a;]]"+alreadyExists;
        if(aec.analyseResult) {
           aec.getPhotoFileProcessorResultGenerator().addGroupData(alreadyExists, baseAttrs);
           }
        aec.statisticSkipped++;
        return;
        } else {
        saveJSON = true; // osm.org does not exist load and save json from osm
        }
      }
      
    def millisSinceLastAccess=System.currentTimeMillis()- aec.lastWebAccess; // time passed since last access of API
    if(millisSinceLastAccess < 1000) {                                       // based on the usage policies of the OSM API
      sleep(1000-millisSinceLastAccess);                                     // we should not call the service more than one time a second
      }
            
    baseAttrs['write to file']= saveJSON;
    baseAttrs['lat']= photoMetadata.getLatitude();
    baseAttrs['lon']= photoMetadata.getLongitude();
        
    def params = [format: 'json',
	                 lat: photoMetadata.getLatitude(),
	                 lon: photoMetadata.getLongitude(),
        'addressdetails': 1,
	   'accept-language': aec.language
		  		        ];
                             
    def url =new URL("https://nominatim.openstreetmap.org/reverse?"+(params.collect { k,v -> "$k=$v" }.join('&')));
        json=new JSONObject(url.getText("UTF-8",
                                        connectTimeout: 5000, 
                                           readTimeout: 10000, 
                                             useCaches: false, 
                                  allowUserInteraction: false, 
                      requestProperties: ['Connection': 'close',
                                          'User-Agent': 'PicApportGroovy/1.0'
                                         ]));
                                   
    aec.lastWebAccess=System.currentTimeMillis();
    
    if(aec.analyseResult) {  
      aec.getPhotoFileProcessorResultGenerator().addGroupData("Result", baseAttrs);
      aec.getPhotoFileProcessorResultGenerator().addGroupData("json", json);
      }
    if(saveJSON) {
      fileToProcess.addAddonMetadata(addonContext, ADDON_TYPE_ID, json);
      if(osmAlreadyExists) {
        aec.statisticReplaced++;
        } else {
        aec.statisticAdded++;
        }
      } else {
      aec.statisticSkipped++;
      }
    }
  
public void stop(IAddonContext addonContext, IAddonExecutionContext aec) {
    def numProblems =aec.currentPhoto - aec.statisticAdded - aec.statisticReplaced - aec.statisticSkipped - aec.statisticNoGeoData - aec.statisticRemoved - aec.statisticQuery;  
    
	def logMessage=" Add-on terminated. Total processed=" + aec.currentPhoto + " Problems=" + numProblems;
	if(numProblems > 0) {
      addonContext.getLogger().logWarningMessage(logMessage);
	  } else {
	  addonContext.getLogger().logInfoMessage(logMessage);
	  }

    // Print out statistic what we have done
    aec.getPhotoFileProcessorResultGenerator()
       .addGroupData(addonContext.i18n("PicApport OpenStreetMap result",    ["de":"PicApport OpenStreetMap Ergebnis"] ), 
                   [(addonContext.i18n("Total processed",                   ["de":"Anzahl verarbeitet"])):              aec.currentPhoto,
                    (addonContext.i18n("Metadata added",                    ["de":"OpenStreetMap Daten hinzugefügt"])): aec.statisticAdded,
                    (addonContext.i18n("Metadata replaced",                 ["de":"OpenStreetMap Daten ersetzt"])):     aec.statisticReplaced,
                    (addonContext.i18n("Nothing to do",                     ["de":"Nichts zu tun"])):                   aec.statisticSkipped,
                    (addonContext.i18n("No geo-coordinates found",          ["de":"Keine Geo-Koordinaten gefunden"])):  aec.statisticNoGeoData,
                    (addonContext.i18n("OpenStreetMap metadata removed",    ["de":"OpenStreetMap Metadata entfernt"])):           aec.statisticRemoved,          
                    (addonContext.i18n("OpenStreetMap metadata from photo", ["de":"OpenStreetMap Metadata aus Foto angezeigt"])): aec.statisticQuery, 
                    (addonContext.i18n("Metadata problems",                 ["de":"Metadaten Probleme"])):                        numProblems
                   ]);
    }

    			
}
