/**
(c) 2020 Contecon Software GmbH Author E. Schreiner

TagYour.Photos Addon

Doku:
https://www.tagyour.photos/

**/

import de.contecon.picapport.IPhotoMetaData;
import de.contecon.picapport.groovy.IAddonContext;
import de.contecon.picapport.groovy.IAddonExecutionContext;
import de.contecon.picapport.groovy.IAddonFileToProcess;
import de.contecon.picapport.groovy.PhotoFileProcessor;


import org.json.JSONArray;
import org.json.JSONObject;

class TagYourPhotos extends PhotoFileProcessor {
  
def ADDON_TYPE_ID = "tagyourphotos.de";
  
def en_Title ='TagYour.Photos (TYP) Automatic photo tagging';
def de_Title ='TagYour.Photos (TYP) Automatisches taggen von Fotos';
    
def en_Options=['Write only when empty',  'Always overwrite',    'no Changes (just test)',        'Remove TagYour.Photos metadata',     'Show TagYour.Photos metadata of photos'];
def de_Options=['Schreibe nur wenn leer', 'Immer überschreiben', 'keine Änderungen (nur testen)', 'TagYour.Photos Metadaten entfernen', 'TagYour.Photos Daten der Fotos anzeigen'];
  
public Map init(IAddonContext addonContext) {
	addonContext.getLogger().logMessage(" Addon loaded. Autor: E. Schreiner (c)2020 Contecon Software GmbH" );

	def meta =  [
                version: '1.0.0', 
				functions: [
						   f1: [
							   name:       en_Title,
							   desc:       'This Addon needs a valid open API-Key\nfrom https://www.tagyour.photos',
							   permission: 'pap:editmeta:photo',
							   
							   parameter: [
                                          mode: [
                                              type:    'select',
                                              label:   'Overwrite',
                                              options: en_Options,
                                              value:   addonContext.getConfigParAsString('mode', '0')
                                              ],
                                          shouldDetectObjects: [
                                              type:   'checkbox',
                                              label:  'Should objects be detected',
                                              value:  addonContext.getConfigParAsBoolean('shouldDetectObjects', true)
                                              ],
                                          shouldDetectLandmarks: [
                                              type:   'checkbox',
                                              label:  'Should landmarks be detected',
                                              value:  addonContext.getConfigParAsBoolean('shouldDetectLandmarks', true)
                                              ],
                                          shouldDetectLocations: [
                                              type:   'checkbox',
                                              label:  'Should detected locations (gps or image)',
                                              value:  addonContext.getConfigParAsBoolean('shouldDetectLocations', true)
                                              ],
                                          probability: [
                                                type: 'range',
                                                label: 'Probability in percent' ,
                                                value: addonContext.getConfigParAsString('probability', '80'),
                                                min:   '1',
                                                max:   '100',
                                                ],
                                          language: [
                                                type:  'text',
                                                label: 'language',
                                                value:  addonContext.getConfigParAsString('language', System.getProperty("user.language")),
                                                permission: 'pap:admin:addon:config'
                                                ],
                                          apikey: [
                                                type:       'text',
                                                label:      'API-Key',
                                                placeholder:'API-Key from www.tagyour.photos',
                                                value:      addonContext.getConfigParAsString('apikey', ''),
                                                permission: 'pap:admin:addon:config'
                                                ],
                                          updateDefaults: [
                                                type:       'checkbox',
                                                label:      'Save current parameter values as defaults',
                                                value:      false,
                                                permission: 'pap:admin:addon:config'
                                                ],                                                
                                          analyseResult: [
                                              type: 'checkbox',
                                              label: 'analyse result',
                                              value:  addonContext.getConfigParAsBoolean('analyseResult', false)
                                              ]
										  ]											  
							   ]								   
						   ],
				i18n: [
					  'de.f1.name':                       de_Title,
					  'de.f1.desc':                       'Dieses Addon benötigt einen gültigen API-Key\nvon https://www.tagyour.photos',
                      'de.f1.mode.label':                 'Überschreiben',
                      'de.f1.mode.options':               de_Options,
					  'de.f1.apikey.label':               'API-Key',
                      'de.f1.shouldDetectObjects.label':  'Objekte erkennen?',
                      'de.f1.shouldDetectLandmarks.label':'Sehenswürdigkeiten erkennen?',
                      'de.f1.shouldDetectLocations.label':'Standort ermitteln (gps oder Bild)',
                      'de.f1.probability.label':          'Wahrscheinlichkeit in Prozent',
                      'de.f1.language.label':             'Sprache',
                      'de.f1.apikey.placeholder':         'API-Key von www.tagyour.photos',
                      'de.f1.updateDefaults.label':       'Aktuelle Parameter als Vorgabe speichern',
					  'de.f1.analyseResult.label':        'Analysiere Ergebnis',
					  ],						
				]
	}

/**
 * Save current parameter values from GUI in config.json in addon directory
 *   
 * author Eric 22.07.2020
 * @param addonContext
 * @param aec
 */
private void handleDefaultParameterUpdate(IAddonContext addonContext, IAddonExecutionContext aec) {
  addonContext.putConfigPar("mode",                 aec.mode);
  addonContext.putConfigPar("shouldDetectObjects",  aec.shouldDetectObjects);
  addonContext.putConfigPar("shouldDetectLandmarks",aec.shouldDetectLandmarks);
  addonContext.putConfigPar("shouldDetectLocations",aec.shouldDetectLocations);
  addonContext.putConfigPar("probability",          aec.probability);
  addonContext.putConfigPar("language",             aec.language);
  addonContext.putConfigPar("apikey",               aec.apikey);
  addonContext.putConfigPar("analyseResult",        aec.analyseResult);
  addonContext.updateConfigFile();
  }
  
   
/*
 * This method is called one Time befor processing of each photo starts
 * use IAddonExecutionContext aec as a map to store values over the livetime of this procedure 
 */
public void start(IAddonContext addonContext, IAddonExecutionContext aec) {
    aec.setShowResults(true); 
    aec.currentPhoto=0; 
    aec.statisticAdded=0;
    aec.statisticReplaced=0;
    aec.statisticSkipped=0;
    aec.statisticRemoved=0;
    aec.statisticQuery=0;

    def titleMap;
                  
    if(aec.updateDefaults) {
      handleDefaultParameterUpdate(addonContext, aec);
      titleMap =["Run-mode":addonContext.i18n("New parameter settings saved.",  ["de":"Neue Parametervorgaben gespeichert."])
                ];
      aec.signalTermination();
      } else {
      titleMap =["Run-mode":addonContext.i18n(en_Options[aec.mode as Integer],  ["de":de_Options[aec.mode as Integer]]),
                (addonContext.i18n("Language",  ["de":"Sprache"])):aec.language
                ];
      }
    aec.getPhotoFileProcessorResultGenerator().addGroupData(addonContext.i18n(en_Title,  ["de":de_Title] ), titleMap);
    }
  
    
/**
 * This method is called for each selected photo
 */
public void processPhotoFile(IAddonContext addonContext, IAddonExecutionContext aec, IAddonFileToProcess fileToProcess) {
    aec.currentPhoto++;
  
    def mode = aec.mode as Integer;
    def typAlreadyExists = fileToProcess.hasAddonMetadata(ADDON_TYPE_ID) as boolean;
    def json = null as JSONObject;
    
    File originalFile = fileToProcess.getOriginalFile();
    def baseAttrs=[:] as LinkedHashMap; // Content of this map will always be visible below the image (Filename etc.)
        baseAttrs['Name']         = originalFile.getName();
        baseAttrs['Path']         = originalFile.getParent();
        baseAttrs['Length']       = originalFile.length();
        baseAttrs['Last Modified']= new Date(originalFile.lastModified());

    if(mode == 3 || mode == 4) { // Remove or Show "osm.org" Metadata call to osm.org is not required
      if(typAlreadyExists) {
        // remove it
        if(mode == 3) { // remove
          fileToProcess.removeAddonMetadata(addonContext, ADDON_TYPE_ID);
          baseAttrs[ADDON_TYPE_ID+' Metadata']= "[[color:#8bc34a;]]Removed";
          aec.statisticRemoved++;
          } else {
          json = fileToProcess.getAddonMetadataAsJSON(ADDON_TYPE_ID);
          baseAttrs[ADDON_TYPE_ID+' Metadata from Photo']= "[[color:#8bc34a;]]Exists";
          aec.statisticQuery++;
          }
        } else {
        // nothing to do
        baseAttrs[ADDON_TYPE_ID+' Metadata']= "[[color:#DD130E;]]"+addonContext.i18n("TagYour.Photos metadata not found in photo",  ["de":"Keine TagYour.Photos Metadaten im Foto gefunden"] );
        aec.statisticSkipped++;
        }
      if(aec.analyseResult || mode ==4) {
         aec.getPhotoFileProcessorResultGenerator().addGroupData("Result", baseAttrs);
         if(null != json) {
           aec.getPhotoFileProcessorResultGenerator().addGroupData("json", json);
           }
         }
      return;
      }
  
    IPhotoMetaData photoMetadata=fileToProcess.getPhotoMetadata();
    
    def saveJSON = (1 == mode) as boolean; // mode 1 = Always overwrite
    
    if(0 == mode) {
      if(typAlreadyExists) {
        def alreadyExists=addonContext.i18n("TagYour.Photos-Data already loaded", ["de":"TagYour.Photos Daten bereits geladen"]);
        baseAttrs[ADDON_TYPE_ID+' Metadata']= "[[color:#8bc34a;]]"+alreadyExists;
        if(aec.analyseResult) {
           aec.getPhotoFileProcessorResultGenerator().addGroupData(alreadyExists, baseAttrs);
           }
        aec.statisticSkipped++;
        return;
        } else {
        saveJSON = true; // osm.org does not exist load and save json from osm
        }
      }
    
        
    def params = [destinationLanguage: aec.language,
                               labels: aec.shouldDetectObjects,
                            landmarks: aec.shouldDetectLandmarks,
                            locations: aec.shouldDetectLocations
                                     ];
    if(aec.shouldDetectLocations) {
      if(photoMetadata.hasGeoCoordinates()) {
         params['latlng']=photoMetadata.getLatitude()+","+photoMetadata.getLongitude();
         baseAttrs['Coordinates']=params['latlng'];
         } else {
         baseAttrs['Coordinates']="[[color:#DD130E;]]"+addonContext.i18n("No GEO-Coordinates", ["de":"Keine Geo-Koordinaten"]);
         }
      }
    def url =new URL("https://srv.tagyour.photos/analyse_new?"+(params.collect { k,v -> "$k=$v" }.join('&')));
    addonContext.getLogger().logDebugMessage(" "+url);
    
    def connection=url.openConnection();
        connection.setDoOutput(true);
        connection.setDoInput(true);
        connection.requestMethod = 'POST';
        connection.setRequestProperty("typ-applicationkey", aec.apikey);
        connection.setRequestProperty("typ-softwareid", "73654129-740c-4202-8e5a-3b944b12dd5a0");
        connection.setRequestProperty("typ-softwareversion", "3.0.0");
        connection.setRequestProperty("Content-Type", "image/jpeg");
        connection.setRequestProperty("Accept", "application/json");
        connection.setRequestProperty("User-Agent", "PicApportGroovy/1.0");
        connection.getOutputStream().withCloseable { os ->
                                     fileToProcess.writeScaledJpgWithoutMetadata(os, 1600, 1600, 0.7f);
                                     }
          
    def successful = connection.responseCode == 200;
    
    if(aec.analyseResult || !successful) {
      def color = successful ? "" : "[[color:#DD130E;]]";
      baseAttrs["Request"]=         color + url;
      baseAttrs["Api-Key"]=         color + aec.apikey;
      baseAttrs["Response Code"]=   color + connection.responseCode;
      baseAttrs["Response Message"]=color + connection.responseMessage;
      baseAttrs["Content Type"]=    color + connection.contentType;
      baseAttrs["Content Encoding"]=color + connection.contentEncoding;
      baseAttrs["Content Length"]=  color + connection.contentLength;
      aec.getPhotoFileProcessorResultGenerator().addGroupData("Result", baseAttrs);
      }
      
    if(successful) {
      StringBuilder responseText = new StringBuilder();
      new BufferedReader(new InputStreamReader(connection.getInputStream(), "utf-8")).withCloseable { br ->
          String responseLine = null;
          while((responseLine = br.readLine()) != null) {
               responseText.append(responseLine.trim());
               }
          }
      def jsonWeb=new JSONObject(responseText.toString());
      
      if(aec.analyseResult) {
         aec.getPhotoFileProcessorResultGenerator().addGroupData("Web Service Result", jsonWeb.toString(2));
         }
         
      def jsonFiltered=filterResult(addonContext, aec, jsonWeb);
      addOptionalLocationData(jsonWeb, "country", jsonFiltered, "country");
      addOptionalLocationData(jsonWeb, "state",   jsonFiltered, "state");
      addOptionalLocationData(jsonWeb, "city",    jsonFiltered, "city");
      addOptionalLocationData(jsonWeb, "district",jsonFiltered, "district");
            
      if(aec.analyseResult) {
        def info=(saveJSON ? ADDON_TYPE_ID : "FILE NOT UPDATED");
        aec.getPhotoFileProcessorResultGenerator().addGroupData("Filtered Result (${info})", jsonFiltered.toString(2));
        }
         
      if(saveJSON) {
        fileToProcess.addAddonMetadata(addonContext, ADDON_TYPE_ID, jsonFiltered); 
        aec.notifySidebarUpdateRequired(); // If the metadata-sidebar is open it should update it's content after execution of the addon
        if(typAlreadyExists) {
          aec.statisticReplaced++;
          } else {
          aec.statisticAdded++;
          }
        } else {
        aec.statisticSkipped++;
        }  
      }   
    }
    
private JSONObject filterResult(IAddonContext addonContext, IAddonExecutionContext aec, JSONObject webResult) {
    def filteredResult = new JSONObject();
    def tags      = new JSONArray();
    def landmarks = new JSONArray();
    def probability = Double.parseDouble(aec.probability) / 100;

    filteredResult.put("version", "Should be provided by WebAPI");
    filterTagsFromArray("tag",     webResult.getJSONObject("tagging").getJSONArray("labels"), tags, probability);
    filterTagsFromArray("landmark",webResult.getJSONObject("tagging").getJSONArray("landmarks"), landmarks, probability)
    
    tags=reorgTags(tags);
  
    filteredResult.put("tags",     tags)
    filteredResult.put("landmarks",landmarks)
    
    return filteredResult;
    }

private void filterTagsFromArray(name, arrayFrom, arrayTo, probability) {
    for(int i = 0; i < arrayFrom.length(); i++) {
       def tag = arrayFrom.getJSONObject(i);
       def ranking = tag.getDouble("ranking");
       if(probability <= ranking) {
         // OK copy result
         arrayTo.put(new JSONObject().put(name, tag.optString("destinationLanguage")).put("probability", ranking));
         }       
      }
    }  
    
private JSONArray reorgTags(tags) {
   for(int i = 0; i < tags.length(); i++) {
      def currentI=tags.getJSONObject(i);
      for(int j = 0; j < tags.length(); j++) {
         if(i != j) {
           def currentJ=tags.getJSONObject(j);
           if(currentI.getString("tag").length() <= currentJ.getString("tag").length()) {
             if(currentJ.getString("tag").startsWith(currentI.getString("tag"))) {
               currentI.put("removed", true);
               }
             }
           }   
         }
      } 
   def tagsReturn = new JSONArray() as JSONArray;
   for(int i = 0; i < tags.length(); i++) {
      if(!tags.getJSONObject(i).has("removed")) {
        tagsReturn.put(tags.getJSONObject(i));
        }  
      }
   return tagsReturn;  
   }
 
private void addOptionalLocationData(JSONObject jsonWeb, String nameIn, JSONObject jsonOut, String nameOut) {
    def locationIn = jsonWeb.getJSONObject("tagging").getJSONObject("location");
    def value=locationIn.optJSONObject(nameIn);
    if(null != value) {
      def newVal=value.optString("destinationLanguage", value.optString("sourceLanguage",null));
      if(null != newVal) {
        jsonOut.put(nameOut, newVal);
        }
      }   
    }
    
/**
 * called after the last photo has been processed use uded to display
 * a summary board with a navigation button      
 */
public void stop(IAddonContext addonContext, IAddonExecutionContext aec) {
    // Print out statistic what we have done
    aec.getPhotoFileProcessorResultGenerator()
       .addGroupData(addonContext.i18n("PicApport TagYour.Photos result",   ["de":"PicApport TagYour.Photos Ergebnis"] ), 
                   [(addonContext.i18n("Total processed",                   ["de":"Anzahl verarbeitet"])):                         aec.currentPhoto,
                    (addonContext.i18n("TagYour.Photos Metadata added",     ["de":"TagYour.Photos Daten hinzugefügt"])):           aec.statisticAdded,
                    (addonContext.i18n("TagYour.Photos Metadata replaced",  ["de":"TagYour.Photos Daten ersetzt"])):               aec.statisticReplaced,
                    (addonContext.i18n("Nothing to do",                     ["de":"Nichts zu tun"])):                              aec.statisticSkipped,
                    (addonContext.i18n("TagYour.Photos metadata removed",   ["de":"TagYour.Photos Metadata entfernt"])):           aec.statisticRemoved,          
                    (addonContext.i18n("TagYour.Photos metadata from photo",["de":"TagYour.Photos Metadata aus Foto angezeigt"])): aec.statisticQuery, 
                    (addonContext.i18n("Problems", ["de":"Probleme"])): (aec.currentPhoto
                                                                        -aec.statisticAdded
                                                                        -aec.statisticReplaced
                                                                        -aec.statisticSkipped
                                                                        -aec.statisticRemoved
                                                                        -aec.statisticQuery)
                   ]);
    }
			
}
